--========================================================--
-- Unseasonal Weather - Climate Event System (Build 42)
-- Con fade-out dinamico segun tipo de evento (Normal/Rare/Ultra)
--========================================================--

if not UW then UW = {} end

------------------------------------------------------------
-- Config por defecto (se sobreescribe con SandboxVars)
------------------------------------------------------------
UW.config = {
    enabled             = true,
    dailyEventChance    = 25,
    minDuration         = 3,
    maxDuration         = 8,
    intensityMultiplier = 1.0,
    rareEventChance     = 5,
    hardcoreWinter      = false,
    hardcoreSummer      = false,
}

UW.originalClimate      = UW.originalClimate or {}
UW._eventsRegistered    = UW._eventsRegistered or false
UW.activeEventName      = UW.activeEventName or nil
UW.eventEndWorldHours   = UW.eventEndWorldHours or nil
UW._scheduledEventDay   = UW._scheduledEventDay or nil
UW._scheduledEventHour  = UW._scheduledEventHour or nil
UW._currentEventType    = UW._currentEventType or nil

------------------------------------------------------------
-- Fade-Out: estados internos
------------------------------------------------------------
UW._fadeActive        = UW._fadeActive or false
UW._fadeProgress      = UW._fadeProgress or 0.0
UW._fadeStartClimate  = UW._fadeStartClimate or nil
UW._fadeDurationHours = UW._fadeDurationHours or 0.33 -- 20 min por defecto

UW.fadeDurations = {
    Normal = 0.50, -- ~30 min (transicion mas suave)
    Rare   = 0.90, -- ~54 min
    Ultra  = 1.30, -- ~78 min
}

------------------------------------------------------------
-- Fade-In: estados internos
------------------------------------------------------------
UW._fadeInActive          = UW._fadeInActive or false
UW._fadeInProgress        = UW._fadeInProgress or 0.0
UW._fadeInDurationHours   = UW._fadeInDurationHours or 0.16 -- 10 min por defecto
UW._fadeInStartClimate    = UW._fadeInStartClimate or nil
UW._fadeInTargetClimate   = UW._fadeInTargetClimate or nil
UW._fadeInStartOverlayFog = UW._fadeInStartOverlayFog or nil
UW._fadeInTargetOverlayFog= UW._fadeInTargetOverlayFog or nil
UW._fadeInStartOverlayCld = UW._fadeInStartOverlayCld or nil
UW._fadeInTargetOverlayCld= UW._fadeInTargetOverlayCld or nil

UW.fadeInDurations = {
    Normal = 0.33, -- ~20 min (mas suavizado)
    Rare   = 0.50, -- ~30 min
    Ultra  = 0.75, -- ~45 min
}

------------------------------------------------------------
-- Helper: ClimateManager
------------------------------------------------------------
local function UW_getClimateManager()
    local world = getWorld()
    if not world then return nil end
    return world:getClimateManager()
end

local function UW_unpack4(t)
    if table and table.unpack then
        return table.unpack(t)
    end
    if unpack then
        return unpack(t)
    end
    return t[1], t[2], t[3], t[4]
end

------------------------------------------------------------
-- Variacion aleatoria +/-20%
------------------------------------------------------------
local function UW_variation(value)
    if not value or value == 0 then return 0 end
    if not ZombRandFloat then
        local factor = 0.8 + (ZombRand(41) / 100.0)
        return value * factor
    end
    return value * ZombRandFloat(0.8, 1.2)
end

------------------------------------------------------------
-- 1. Tabla de eventos por temporada (mas variedad)
--  temperature = delta respecto a la actual
--  wind/clouds = delta respecto al valor actual
--  rain/fog    = valor absoluto 0-1
------------------------------------------------------------
UW.EventsBySeason = {
    spring = {
        { name = "LightDrizzle",   data = { rain = 0.3,  temperature =  1, clouds = 0.2 } },
        { name = "WindyDay",       data = { wind = 0.5,  clouds      =  0.3 } },
        { name = "FoggyMorning",   data = { fog  = 0.6,  clouds      =  0.4 } },
        { name = "SpringStorm",    data = { rain = 0.7,  temperature =  2, wind = 0.5, clouds = 0.6 } },
        { name = "ColdDrizzle",    data = { rain = 0.4,  temperature = -3, clouds = 0.5 } },
        { name = "ClearSkies",     data = { temperature =  2, wind = -0.2, clouds = -0.3 } },
        { name = "HumidDay",       data = { rain = 0.2,  fog = 0.3, temperature = 3, clouds = 0.4 } },
    },
    summer = {
        { name = "HeatWave",       data = { temperature =  10, wind = 0.15, clouds = -0.1, fog = 0.1 } },
        { name = "TropicalRain",   data = { rain = 0.8,  temperature =  3, clouds = 0.4 } },
        { name = "CalmDay",        data = { wind = -0.4, clouds      = -0.2 } },
        { name = "DryHeat",        data = { temperature =  12, wind = 0.12, clouds = -0.3, fog = 0.05 } },
        { name = "SummerStorm",    data = { rain = 0.9,  temperature =  1, wind = 0.6, clouds = 0.7 } },
        { name = "HeavyDownpour",  data = { rain = 1.0,  temperature =  0, wind = 0.3, clouds = 0.8 } },
        { name = "DustyWinds",     data = { temperature =  6, wind = 0.7, clouds = -0.1 } },
    },
    autumn = {
        { name = "ColdFront",      data = { temperature = -5, wind = 0.4, clouds = 0.3 } },
        { name = "Drizzle",        data = { rain = 0.3,  clouds = 0.3 } },
        { name = "FoggyEvening",   data = { fog  = 0.7,  clouds = 0.4 } },
        { name = "AutumnStorm",    data = { rain = 0.8,  temperature = -1, wind = 0.5, clouds = 0.7 } },
        { name = "WarmAutumnDay",  data = { temperature =  3, wind = -0.2, clouds = -0.2 } },
        { name = "WindyShowers",   data = { rain = 0.5,  wind = 0.6, clouds = 0.5 } },
        { name = "MistyDawn",      data = { fog  = 0.5,  temperature = -2, clouds = 0.3 } },
    },
    winter = {
        { name = "Snowstorm",      data = { rain = 0.9,  temperature = -8, wind = 0.5, clouds = 0.5 } },
        { name = "Blizzard",       data = { rain = 1.0,  temperature = -10, wind = 0.7, fog = 0.6, clouds = 0.6 } },
        { name = "ClearColdDay",   data = { temperature = -6, wind = -0.3, clouds = -0.2 } },
        { name = "FreezingDrizzle",data = { rain = 0.4,  temperature = -5, clouds = 0.3 } },
        { name = "DeepFreeze",     data = { temperature = -12, wind = 0.4, clouds = 0.1 } },
        { name = "HeavySnow",      data = { rain = 1.0,  temperature = -9, wind = 0.4, clouds = 0.7 } },
        { name = "WindySnow",      data = { rain = 0.7,  temperature = -7, wind = 0.6, clouds = 0.5 } },
    }
}

------------------------------------------------------------
-- Eventos hardcore (solo se usan si se activan en sandbox)
------------------------------------------------------------
UW.HardcoreEvents = {
    summer = {
        { name = "InfernalHeatwave", data = { temperature = 18, wind = 0.05, clouds = -0.2, fog = 0.0, rain = 0.0 } },
        { name = "ScorchingDrought", data = { temperature = 16, wind = 0.08, clouds = -0.3, fog = 0.0, rain = 0.0 } },
    },
    winter = {
        { name = "PolarVortex",   data = { temperature = -20, wind = 0.75, fog = 0.35, clouds = 0.6, rain = 0.0 } },
        { name = "GlacialNight",  data = { temperature = -18, wind = 0.65, fog = 0.25, clouds = 0.4, rain = 0.0 } },
    }
}

------------------------------------------------------------
-- Merge helper para añadir eventos hardcore sin mutar tabla
------------------------------------------------------------
local function UW_mergeEvents(base, extra)
    if not extra or #extra == 0 then return base end
    if not base or #base == 0 then return extra end

    local merged = {}
    for i = 1, #base do
        merged[#merged + 1] = base[i]
    end
    for i = 1, #extra do
        merged[#merged + 1] = extra[i]
    end
    return merged
end

------------------------------------------------------------
-- Eventos raros globales (pueden salir en cualquier estacion)
------------------------------------------------------------
UW.RareEvents = {
    -- 1 .Tormenta de niebla super densa
    {
        name = "SuperFog",
        data = {
            fog         = 1.0,
            clouds      = 0.8,
            temperature = -1,
            wind        = -0.2,
            rain        = 0.0,
        }
    },

    -- 2. Dia ultra despejado y "cristalino"
    {
        name = "CrystalClearDay",
        data = {
            fog         = 0.0,
            clouds      = -0.8,
            wind        = -0.3,
            temperature = 2,
            rain        = 0.0,
        }
    },

    -- 3. Moonlit Clarity - Noche ultra brillante
    {
        name = "MoonlitClarity",
        data = {
            temperature = -2,
            fog         = 0.00,
            clouds      = -0.60,
            wind        = -0.10,
            rain        = 0.0,
        }
    },

    -- 4. Blood Sky - Cielos rojizos densos
    {
        name = "BloodSky",
        data = {
            temperature = 1,
            fog         = 0.25,
            clouds      = 0.65,
            wind        = 0.20,
            rain        = 0.0,
        }
    },

    -- 5. Static Storm - Tormenta sin lluvia
    {
        name = "StaticStorm",
        data = {
            temperature = -1,
            fog         = 0.10,
            clouds      = 0.85,
            wind        = 0.65,
            rain        = 0.0,
        }
    },

    -- 6. Ashfall / Dustfall - Niebla de ceniza suave
    {
        name = "Ashfall",
        data = {
            temperature = -1,
            fog         = 0.40,
            clouds      = 0.50,
            wind        = 0.25,
            rain        = 0.0,
        }
    },

    -- 7. Frozen Dawn - Amanecer congelado
    {
        name = "FrozenDawn",
        data = {
            temperature = -8,
            fog         = 0.12,
            clouds      = 0.20,
            wind        = 0.35,
            rain        = 0.0,
        }
    },

    -- 8. Silver Skies - Cielo plateado limpio y brillante
    {
        name = "SilverSkies",
        data = {
            temperature = 1,
            fog         = 0.05,
            clouds      = 0.30,
            wind        = -0.10,
            rain        = 0.0,
        }
    },
}

UW.UltraRareEvents = {
    -- 1. Solar Hellburst - Tormenta solar rojiza extrema (Dune)
    {
        name = "SolarHellburst",
        data = {
            temperature = 10,
            fog         = 0.35,
            clouds      = 0.7,
            wind        = -0.5,
            rain        = 0.0,
        }
    },

    -- 2. White Death - Amanecer polar azul extremo
    {
        name = "WhiteDeath",
        data = {
            temperature = -14,
            fog         = 0.18,
            clouds      = 0.15,
            wind        = 0.45,
            rain        = 0.0,
        }
    },
}

------------------------------------------------------------
-- 2. Leer SandboxVars
------------------------------------------------------------
local function UW_ApplySandbox()
    if not SandboxVars or not SandboxVars.UnseasonalWeather then return end
    local sv = SandboxVars.UnseasonalWeather

    if sv.Enabled ~= nil then
        UW.config.enabled = sv.Enabled
    end
    if sv.DailyEventChance ~= nil then
        UW.config.dailyEventChance = tonumber(sv.DailyEventChance) or UW.config.dailyEventChance
    end
    if sv.MinDuration ~= nil then
        UW.config.minDuration = tonumber(sv.MinDuration) or UW.config.minDuration
    end
    if sv.MaxDuration ~= nil then
        UW.config.maxDuration = tonumber(sv.MaxDuration) or UW.config.maxDuration
    end
    if sv.IntensityMultiplier ~= nil then
        UW.config.intensityMultiplier = tonumber(sv.IntensityMultiplier) or UW.config.intensityMultiplier
    end
    if sv.HardcoreWinterTemps ~= nil then
        UW.config.hardcoreWinter = sv.HardcoreWinterTemps == true
    end
    if sv.HardcoreSummerTemps ~= nil then
        UW.config.hardcoreSummer = sv.HardcoreSummerTemps == true
    end

    if UW.config.maxDuration < UW.config.minDuration then
        UW.config.maxDuration = UW.config.minDuration
    end

    print(string.format(
        "[UW] Sandbox aplicado -> Enabled=%s Chance=%d%% Duracion=%d-%d h Intensidad=%.2f HardcoreInv=%s HardcoreVer=%s",
        tostring(UW.config.enabled),
        UW.config.dailyEventChance,
        UW.config.minDuration,
        UW.config.maxDuration,
        UW.config.intensityMultiplier,
        tostring(UW.config.hardcoreWinter),
        tostring(UW.config.hardcoreSummer)
    ))
end

------------------------------------------------------------
-- 3. Obtener temporada actual
------------------------------------------------------------
function UW.getCurrentSeason()
    local cm = UW_getClimateManager()
    if not cm then return "spring" end

    local seasonName = cm:getSeasonName() or ""
    seasonName = string.lower(seasonName)

    if seasonName:find("spring", 1, true) then
        return "spring"
    elseif seasonName:find("summer", 1, true) then
        return "summer"
    elseif seasonName:find("autumn", 1, true) or seasonName:find("fall", 1, true) then
        return "autumn"
    elseif seasonName:find("winter", 1, true) then
        return "winter"
    end

    return "spring"
end

------------------------------------------------------------
-- 4. Escoger evento aleatorio segun temporada
------------------------------------------------------------
function UW.pickRandomEvent()
    local season = UW.getCurrentSeason()
    local events = UW.EventsBySeason[season]

    if season == "summer" and UW.config.hardcoreSummer then
        events = UW_mergeEvents(events, UW.HardcoreEvents.summer)
    elseif season == "winter" and UW.config.hardcoreWinter then
        events = UW_mergeEvents(events, UW.HardcoreEvents.winter)
    end

    if not events or #events == 0 then return nil end

    -- Raros globales
    if UW.config.rareEventChance
    and UW.config.rareEventChance > 0
    and UW.RareEvents
    and #UW.RareEvents > 0 then

        local roll = ZombRand(100)
        if roll < UW.config.rareEventChance then
            local rIndex = ZombRand(#UW.RareEvents) + 1
            local rare   = UW.RareEvents[rIndex]

            print(string.format(
                "[UW] >> Evento raro activado (%s) - tirada=%d / umbral=%d",
                rare.name or "Unknown",
                roll,
                UW.config.rareEventChance
            ))

            return rare
        end
    end

    -- Ultra raros
    if UW.UltraRareEvents and #UW.UltraRareEvents > 0 then
        local rollUR = ZombRand(1000) -- 0.1% probabilidad
        if rollUR == 0 then
            local idx = ZombRand(#UW.UltraRareEvents) + 1
            local ultra = UW.UltraRareEvents[idx]

            print(string.format(
                "[UW] >> Evento ULTRA raro activado: %s",
                ultra.name
            ))
            return ultra
        end
    end

    -- Evento de temporada
    local idx = ZombRand(#events) + 1
    return events[idx]
end

------------------------------------------------------------
-- Debug: forzar un evento UltraRaro
------------------------------------------------------------
function UW.UltraRareEvent(name)
    if not UW.UltraRareEvents or #UW.UltraRareEvents == 0 then
        print("[UW] No hay eventos UltraRaros definidos.")
        return
    end

    local chosen = nil
    if name then
        for _, ev in ipairs(UW.UltraRareEvents) do
            if ev.name == name then
                chosen = ev
                break
            end
        end

        if not chosen then
            print("[UW] UltraRareEvent '" .. tostring(name) .. "' no encontrado. Usando uno aleatorio.")
        end
    end

    if not chosen then
        local idx = ZombRand(#UW.UltraRareEvents) + 1
        chosen = UW.UltraRareEvents[idx]
    end

    local oldPick = UW.pickRandomEvent
    UW.pickRandomEvent = function()
        return chosen
    end

    print(string.format(
        "[UW] >> Forzando UltraRareEvent de debug: %s",
        tostring(chosen.name or "UnknownUltra")
    ))

    local ok, err = pcall(UW.startEvent)
    if not ok then
        print("[UW] Error al iniciar UltraRareEvent: " .. tostring(err))
    end

    UW.pickRandomEvent = oldPick
end

------------------------------------------------------------
-- 5. Guardar clima original
------------------------------------------------------------
function UW.captureOriginalClimate()
    local cm = UW_getClimateManager()
    if not cm then
        print("[UW] ERROR: ClimateManager no disponible al capturar clima.")
        return UW.originalClimate
    end

    local t = {
        temperature = cm:getTemperature()            or 0.0,
        rain        = cm:getPrecipitationIntensity() or 0.0,
        fog         = cm:getFogIntensity()           or 0.0,
        wind        = cm:getWindIntensity()          or 0.0,
        clouds      = cm:getCloudIntensity()         or 0.0
    }

    UW.originalClimate = t

    print(string.format(
        "[UW] Clima capturado -> Temp %.2f | Rain %.2f | Fog %.2f | Wind %.2f | Clouds %.2f",
        t.temperature, t.rain, t.fog, t.wind, t.clouds
    ))

    return t
end

------------------------------------------------------------
-- 6. Iniciar evento (con deteccion de tipo para fade-out)
------------------------------------------------------------
function UW.startEvent(eventName)
    if not UW.config.enabled then
        print("[UW] Evento cancelado: desactivado en Sandbox.")
        return
    end

    local cm = UW_getClimateManager()
    if not cm then
        print("[UW] ClimateManager no esta listo, cancelando inicio del evento.")
        return
    end

    if UW.eventEndWorldHours then
        print("[UW] Ya hay un evento activo, se ignora nueva activacion.")
        return
    end

    local chosen = UW.pickRandomEvent()

    if not chosen or not chosen.data then
        print("[UW] No se encontro evento climatico valido o sin datos.")
        return
    end

    -- Determinar tipo (Normal/Rare/Ultra) para el fade-out
    local evType = "Normal"
    for _, ev in ipairs(UW.RareEvents or {}) do
        if ev.name == chosen.name then
            evType = "Rare"
            break
        end
    end
    for _, ev in ipairs(UW.UltraRareEvents or {}) do
        if ev.name == chosen.name then
            evType = "Ultra"
            break
        end
    end
    UW._currentEventType = evType

    local src        = chosen.data or {}
    local data       = {
        temperature = src.temperature,
        rain        = src.rain,
        fog         = src.fog,
        wind        = src.wind,
        clouds      = src.clouds,
    }

    local original   = UW.captureOriginalClimate()
    local intensity  = UW.config.intensityMultiplier or 1.0

    data.temperature = tonumber(data.temperature) or 0
    data.rain        = tonumber(data.rain)        or 0
    data.fog         = tonumber(data.fog)         or 0
    data.wind        = tonumber(data.wind)        or 0
    data.clouds      = tonumber(data.clouds)      or 0

    data.temperature = UW_variation(data.temperature * intensity)
    data.rain        = UW_variation(data.rain * intensity)
    data.fog         = UW_variation(data.fog * intensity)
    data.wind        = UW_variation(data.wind * intensity)
    data.clouds      = UW_variation(data.clouds * intensity)

    local baseTemp   = tonumber(original.temperature) or 0
    local baseWind   = tonumber(original.wind)        or 0
    local baseClouds = tonumber(original.clouds)      or 0

    local target = {}
    local overlayFogTarget, overlayCloudTarget
    local season = UW.getCurrentSeason()

    --------------------------------------------------------
    -- Overrides (solo se guardan; se aplican con fade-in)
    --------------------------------------------------------

    -- Temperatura (delta)
    if data.temperature ~= 0 then
        local newTemp = baseTemp + data.temperature
        local finalTemp = newTemp

        if season == "winter" and data.rain > 0 then
            finalTemp = math.min(newTemp, -1.0)
        end

        target.temperature = finalTemp

        print(string.format(
            "[UW] Override temperatura (target): base=%.2f delta=%.2f new=%.2f (final=%.2f)",
            baseTemp, data.temperature, newTemp, finalTemp
        ))
    end

    -- Lluvia (absoluto 0-1)
    if data.rain ~= 0 then
        local rain = math.min(1.0, math.max(0.0, data.rain))
        target.rain = rain
        print(string.format("[UW] Override lluvia (target): %.2f", rain))
    end

    -- Si es invierno y llueve, usar nieve en lugar de lluvia
    if season == "winter" and data.rain > 0 then
        target.snow = data.rain
        print("[UW] Forzando nieve en invierno (rain->snow), target=" .. tostring(data.rain))
    end

    -- Niebla (absoluto 0-1)
    if data.fog ~= 0 then
        local fog = math.min(1.0, math.max(0.0, data.fog))
        target.fog = fog
        print(string.format("[UW] Override niebla (target): %.2f", fog))
    end

    -- Viento (delta)
    if data.wind ~= 0 then
        local newWind = math.min(1.0, math.max(0.0, baseWind + data.wind))
        target.wind = newWind
        print(string.format(
            "[UW] Override viento (target): base=%.2f delta=%.2f new=%.2f",
            baseWind, data.wind, newWind
        ))
    end

    -- Nubes (delta)
    if data.clouds ~= 0 then
        local newClouds = math.min(1.0, math.max(0.0, baseClouds + data.clouds))
        target.clouds = newClouds
        print(string.format(
            "[UW] Override nubes (target): base=%.2f delta=%.2f new=%.2f",
            baseClouds, data.clouds, newClouds
        ))
    end

    ------------------------------------------------------------
    -- OVERLAY (guardado para fade-in)
    ------------------------------------------------------------
    if chosen.name == "SolarHellburst" then
        overlayFogTarget   = { 1.0, 0.35, 0.05, 0.9 }
        overlayCloudTarget = { 1.0, 0.35, 0.05, 0.7 }
        print("[UW] >> Overlay target: Solar Hellburst")
    end

    if chosen.name == "WhiteDeath" then
        overlayFogTarget   = { 0.20, 0.35, 1.0, 0.9 }
        overlayCloudTarget = { 0.35, 0.45, 1.0, 0.75 }
        print("[UW] Overlay target: White Death")
    end

    local hasTargets = target.temperature ~= nil
        or target.rain ~= nil
        or target.fog ~= nil
        or target.wind ~= nil
        or target.clouds ~= nil
        or target.snow ~= nil
        or overlayFogTarget ~= nil
        or overlayCloudTarget ~= nil

    -- Duracion del evento (total = fade-in + cuerpo + fade-out)
    local gt           = getGameTime()
    local worldHours   = gt:getWorldAgeHours()
    local fadeInDur    = UW.fadeInDurations[evType] or UW.fadeInDurations.Normal
    local fadeOutDur   = UW.fadeDurations[evType]   or UW.fadeDurations.Normal
    local totalTarget  = ZombRand(UW.config.minDuration, UW.config.maxDuration + 1)
    local bodyDuration = totalTarget - fadeInDur - fadeOutDur
    if bodyDuration < 0.25 then bodyDuration = 0.25 end

    UW.eventEndWorldHours = worldHours + bodyDuration
    UW.activeEventName    = chosen.name or (eventName or "RandomEvent")

    if hasTargets then
        UW.startFadeIn(target, overlayFogTarget, overlayCloudTarget)
    end

    print(string.format(
        "[UW] >> Evento iniciado: %s | tipo=%s | Temp %.2f | Rain %.2f | Fog %.2f | Wind %.2f | Clouds %.2f | Duracion total: %.1f h (fadeIn %.2f + cuerpo %.2f + fadeOut %.2f)",
        UW.activeEventName, evType, data.temperature, data.rain, data.fog, data.wind, data.clouds, (fadeInDur + bodyDuration + fadeOutDur), fadeInDur, bodyDuration, fadeOutDur
    ))
end

------------------------------------------------------------
-- 7. Iniciar fade-in suave
------------------------------------------------------------
function UW.startFadeIn(target, overlayFogTarget, overlayCloudTarget)
    if UW._fadeInActive then return end

    local cm = UW_getClimateManager()
    if not cm then
        print("[UW] No se pudo iniciar fade-in (ClimateManager no disponible).")
        return
    end

    UW._fadeInTargetClimate    = target or {}
    UW._fadeInTargetOverlayFog = overlayFogTarget
    UW._fadeInTargetOverlayCld = overlayCloudTarget

    local orig = UW.originalClimate or UW.captureOriginalClimate() or {}
    local tempF = cm:getClimateFloat(4)
    local rainF = cm:getClimateFloat(3)
    local fogF  = cm:getClimateFloat(5)
    local windF = cm:getClimateFloat(6)
    local cldF  = cm:getClimateFloat(7)
    local snowF = cm:getClimateFloat(8)

    UW._fadeInStartClimate = {
        temperature = orig.temperature or 0.0,
        rain        = orig.rain or 0.0,
        fog         = orig.fog or 0.0,
        wind        = orig.wind or 0.0,
        clouds      = orig.clouds or 0.0,
        snow        = (snowF and snowF:getOverride()) or 0.0,
    }

    local fogColor = cm:getClimateColor(0)
    local cloudColor = cm:getClimateColor(1)

    UW._fadeInStartOverlayFog = (fogColor and fogColor.isEnableOverride and fogColor:isEnableOverride() and fogColor.getR)
        and { fogColor:getR(), fogColor:getG(), fogColor:getB(), fogColor:getA() }
        or { 1.0, 1.0, 1.0, 0.0 }

    UW._fadeInStartOverlayCld = (cloudColor and cloudColor.isEnableOverride and cloudColor:isEnableOverride() and cloudColor.getR)
        and { cloudColor:getR(), cloudColor:getG(), cloudColor:getB(), cloudColor:getA() }
        or { 1.0, 1.0, 1.0, 0.0 }

    local evType = UW._currentEventType or "Normal"
    UW._fadeInDurationHours = UW.fadeInDurations[evType] or UW.fadeInDurations.Normal
    UW._fadeInProgress = 0
    UW._fadeInActive   = true

    -- Habilitar overrides con el valor inicial (clima natural)
    if target.temperature and tempF then
        tempF:setEnableOverride(true)
        tempF:setOverride(UW._fadeInStartClimate.temperature, 1.0)
    end
    if target.rain and rainF then
        rainF:setEnableOverride(true)
        rainF:setOverride(UW._fadeInStartClimate.rain, 1.0)
    end
    if target.fog and fogF then
        fogF:setEnableOverride(true)
        fogF:setOverride(UW._fadeInStartClimate.fog, 1.0)
    end
    if target.wind and windF then
        windF:setEnableOverride(true)
        windF:setOverride(UW._fadeInStartClimate.wind, 1.0)
    end
    if target.clouds and cldF then
        cldF:setEnableOverride(true)
        cldF:setOverride(UW._fadeInStartClimate.clouds, 1.0)
    end
    if target.snow and snowF then
        snowF:setEnableOverride(true)
        snowF:setOverride(UW._fadeInStartClimate.snow, 1.0)
    end

    if overlayFogTarget and fogColor then
        fogColor:setEnableOverride(true)
        fogColor:setOverride(UW_unpack4(UW._fadeInStartOverlayFog))
    end
    if overlayCloudTarget and cloudColor then
        cloudColor:setEnableOverride(true)
        cloudColor:setOverride(UW_unpack4(UW._fadeInStartOverlayCld))
    end

    print(string.format("[UW] >> Iniciado fade-in tipo=%s duracion=%.2f h", evType, UW._fadeInDurationHours))
end

------------------------------------------------------------
-- 8. Proceso de fade-in (cada 10 min)
------------------------------------------------------------
local function UW_processFadeIn()
    if not UW._fadeInActive then return end

    local cm = UW_getClimateManager()
    if not cm then return end

    local start = UW._fadeInStartClimate
    local target = UW._fadeInTargetClimate
    if not start or not target then
        UW._fadeInActive = false
        return
    end

    local step = (10/60) / (UW._fadeInDurationHours > 0 and UW._fadeInDurationHours or 0.16)
    UW._fadeInProgress = UW._fadeInProgress + step

    local t = math.min(UW._fadeInProgress, 1.0)
    local function lerp(a, b)
        a = tonumber(a) or 0
        b = tonumber(b) or 0
        return a + (b - a) * t
    end

    local tempF = cm:getClimateFloat(4)
    if target.temperature and tempF then
        tempF:setOverride(lerp(start.temperature, target.temperature), 1.0)
    end

    local rainF = cm:getClimateFloat(3)
    if target.rain and rainF then
        rainF:setOverride(lerp(start.rain, target.rain), 1.0)
    end

    local fogF = cm:getClimateFloat(5)
    if target.fog and fogF then
        fogF:setOverride(lerp(start.fog, target.fog), 1.0)
    end

    local windF = cm:getClimateFloat(6)
    if target.wind and windF then
        windF:setOverride(lerp(start.wind, target.wind), 1.0)
    end

    local cldF = cm:getClimateFloat(7)
    if target.clouds and cldF then
        cldF:setOverride(lerp(start.clouds, target.clouds), 1.0)
    end

    local snowF = cm:getClimateFloat(8)
    if target.snow and snowF then
        snowF:setOverride(lerp(start.snow, target.snow), 1.0)
    end

    local fogColor = cm:getClimateColor(0)
    local fogTarget = UW._fadeInTargetOverlayFog
    if fogTarget and fogColor and fogColor.setOverride then
        local s = UW._fadeInStartOverlayFog or { 1, 1, 1, 0 }
        fogColor:setOverride(
            lerp(s[1], fogTarget[1]),
            lerp(s[2], fogTarget[2]),
            lerp(s[3], fogTarget[3]),
            lerp(s[4], fogTarget[4])
        )
    end

    local cloudColor = cm:getClimateColor(1)
    local cldTarget = UW._fadeInTargetOverlayCld
    if cldTarget and cloudColor and cloudColor.setOverride then
        local s = UW._fadeInStartOverlayCld or { 1, 1, 1, 0 }
        cloudColor:setOverride(
            lerp(s[1], cldTarget[1]),
            lerp(s[2], cldTarget[2]),
            lerp(s[3], cldTarget[3]),
            lerp(s[4], cldTarget[4])
        )
    end

    if t >= 1.0 then
        UW._fadeInActive = false
        UW._fadeInStartClimate    = nil
        UW._fadeInTargetClimate   = nil
        UW._fadeInStartOverlayFog = nil
        UW._fadeInTargetOverlayFog= nil
        UW._fadeInStartOverlayCld = nil
        UW._fadeInTargetOverlayCld= nil
        print("[UW] >> Fade-in completado.")
    end
end

------------------------------------------------------------
-- 9. Iniciar fade-out suave
------------------------------------------------------------
function UW.startFadeOut()
    if UW._fadeActive then return end

    local cm = UW_getClimateManager()
    if not cm then return UW.endEvent() end

    -- cancelar un fade-in en progreso si lo hubiera
    UW._fadeInActive = false

    local tempF = cm:getClimateFloat(4)
    local rainF = cm:getClimateFloat(3)
    local fogF  = cm:getClimateFloat(5)
    local windF = cm:getClimateFloat(6)
    local cldF  = cm:getClimateFloat(7)

    UW._fadeStartClimate = {
        temperature = tempF and tempF:getOverride() or 0.0,
        rain        = rainF and rainF:getOverride() or 0.0,
        fog         = fogF  and fogF:getOverride()  or 0.0,
        wind        = windF and windF:getOverride() or 0.0,
        clouds      = cldF  and cldF:getOverride()  or 0.0,
    }

    local t = UW._currentEventType or "Normal"
    UW._fadeDurationHours = UW.fadeDurations[t] or UW.fadeDurations.Normal

    UW._fadeProgress = 0
    UW._fadeActive   = true

    print("[UW] >> Iniciado fade-out para tipo " .. t)
end

------------------------------------------------------------
-- 10. Proceso de fade-out (cada 10 min)
------------------------------------------------------------
local function UW_processFadeOut()
    if not UW._fadeActive then return end

    local cm = UW_getClimateManager()
    if not cm then return end

    local orig = UW.originalClimate
    local cur  = UW._fadeStartClimate
    if not orig or not cur then
        UW._fadeActive = false
        UW.endEvent()
        return
    end

    local step = (10/60) / (UW._fadeDurationHours > 0 and UW._fadeDurationHours or 0.33)
    UW._fadeProgress = UW._fadeProgress + step

    local t = math.min(UW._fadeProgress, 1.0)
    local function lerp(a, b)
        a = tonumber(a) or 0
        b = tonumber(b) or 0
        return a + (b - a) * t
    end

    local tempF = cm:getClimateFloat(4)
    if tempF then tempF:setOverride(lerp(cur.temperature, orig.temperature), 1.0) end

    local rainF = cm:getClimateFloat(3)
    if rainF then rainF:setOverride(lerp(cur.rain, orig.rain), 1.0) end

    local fogF = cm:getClimateFloat(5)
    if fogF then fogF:setOverride(lerp(cur.fog, orig.fog), 1.0) end

    local windF = cm:getClimateFloat(6)
    if windF then windF:setOverride(lerp(cur.wind, orig.wind), 1.0) end

    local cldF = cm:getClimateFloat(7)
    if cldF then cldF:setOverride(lerp(cur.clouds, orig.clouds), 1.0) end

    -- Fade de overlays si estaban activos
    local fogC = cm:getClimateColor(0)
    local fogHasChannels = fogC and fogC.getR and fogC.getG and fogC.getB and fogC.getA
    if fogC and fogC.isEnableOverride and fogC:isEnableOverride() and fogHasChannels then
        fogC:setOverride(
            lerp(fogC:getR(), 1.0),
            lerp(fogC:getG(), 1.0),
            lerp(fogC:getB(), 1.0),
            lerp(fogC:getA(), 0.0)
        )
    elseif fogC and fogC.isEnableOverride then
        fogC:setEnableOverride(false)
    end

    local cldC = cm:getClimateColor(1)
    local cldHasChannels = cldC and cldC.getR and cldC.getG and cldC.getB and cldC.getA
    if cldC and cldC.isEnableOverride and cldC:isEnableOverride() and cldHasChannels then
        cldC:setOverride(
            lerp(cldC:getR(), 1.0),
            lerp(cldC:getG(), 1.0),
            lerp(cldC:getB(), 1.0),
            lerp(cldC:getA(), 0.0)
        )
    elseif cldC and cldC.isEnableOverride then
        cldC:setEnableOverride(false)
    end

    if t >= 1.0 then
        print("[UW] >> Fade-out completado. Restaurando clima natural.")
        UW._fadeActive = false
        UW.endEvent()
    end
end

------------------------------------------------------------
-- 11. Terminar evento
------------------------------------------------------------
function UW.endEvent()
    local cm = UW_getClimateManager()
    if not cm then
        print("[UW] No se pudo restaurar clima, ClimateManager no disponible.")
        UW.eventEndWorldHours = nil
        UW.activeEventName    = nil
        UW._currentEventType  = nil
        UW._fadeActive        = false
        return
    end

    local snow = cm:getClimateFloat(8)
    if snow then snow:setEnableOverride(false) end

    cm:getClimateFloat(3):setEnableOverride(false)
    cm:getClimateFloat(4):setEnableOverride(false)
    cm:getClimateFloat(5):setEnableOverride(false)
    cm:getClimateFloat(6):setEnableOverride(false)
    cm:getClimateFloat(7):setEnableOverride(false)

    local fogColor = cm:getClimateColor(0)
    if fogColor then fogColor:setEnableOverride(false) end

    local cloudColor = cm:getClimateColor(1)
    if cloudColor then cloudColor:setEnableOverride(false) end

    UW._fadeActive        = false
    UW._fadeStartClimate  = nil
    UW._fadeProgress      = 0
    UW._fadeDurationHours = UW.fadeDurations.Normal
    UW._currentEventType  = nil
    UW._fadeInActive          = false
    UW._fadeInStartClimate    = nil
    UW._fadeInTargetClimate   = nil
    UW._fadeInStartOverlayFog = nil
    UW._fadeInTargetOverlayFog= nil
    UW._fadeInStartOverlayCld = nil
    UW._fadeInTargetOverlayCld= nil
    UW._fadeInProgress        = 0
    UW._fadeInDurationHours   = UW.fadeInDurations.Normal

    UW.eventEndWorldHours = nil
    UW.activeEventName    = nil

    print("[UW] Clima restaurado a condiciones naturales.")

    -- Pensamiento de radio: contar evento relevante
    if UW.RadioAwareness and UW.RadioAwareness.OnClimateEventExperienced then
        UW.RadioAwareness.OnClimateEventExperienced()
    end
end

------------------------------------------------------------
-- 12. Bucle EveryTenMinutes (con fade integrado)
------------------------------------------------------------
function UW.checkForEvent()
    if not UW.config.enabled then return end

    local cm = UW_getClimateManager()
    if not cm then return end

    local gt         = getGameTime()
    local worldHours = gt:getWorldAgeHours()
    local hour       = gt:getHour()
    local dayIndex   = math.floor(worldHours / 24)

    if not UW._scheduledEventDay or UW._scheduledEventDay ~= dayIndex then
        UW._scheduledEventDay  = dayIndex
        UW._scheduledEventHour = ZombRand(24)
        print(string.format(
            "[UW] Nueva hora diaria programada para evento: %02d:00 (dia=%d)",
            UW._scheduledEventHour, dayIndex
        ))
    end

    if UW._fadeInActive then
        UW_processFadeIn()
        return
    end

    if UW.eventEndWorldHours and worldHours >= UW.eventEndWorldHours and not UW._fadeActive and not UW._fadeInActive then
        print("[UW] >> Duracion de evento expirada, iniciando fade-out.")
        UW.startFadeOut()
    end

    if UW._fadeActive then
        UW_processFadeOut()
        return
    end

    if UW.eventEndWorldHours then
        return
    end

    local targetHour = UW._scheduledEventHour or 12

    -- Si hay evento vanilla activo, mover la hora programada (evita solapes).
    if cm.isWeatherPeriodActive and cm:isWeatherPeriodActive() then
        local newHour = (hour + 1) % 24
        if newHour <= hour then
            UW._scheduledEventDay = dayIndex + 1
        else
            UW._scheduledEventDay = dayIndex
        end
        UW._scheduledEventHour = newHour
        print(string.format("[UW] Clima vanilla activo, se reprograma a %02d:00 (dia=%d)", UW._scheduledEventHour, UW._scheduledEventDay))
        return
    end

    if hour == targetHour then
        local roll = ZombRand(100)
        print(string.format(
            "[UW] Chequeo de evento diario a las %02dh. Tirada=%d, Umbral=%d (horaObjetivo=%02d)",
            hour, roll, UW.config.dailyEventChance, targetHour
        ))

        if roll < UW.config.dailyEventChance then
            UW.startEvent()
        end
    end
end

------------------------------------------------------------
-- 13. Registro de eventos
------------------------------------------------------------
local function UW_RegisterEvents()
    if UW._eventsRegistered then return end
    if not Events or not Events.EveryTenMinutes then
        print("[UW] EveryTenMinutes no disponible al registrar.")
        return
    end

    Events.EveryTenMinutes.Add(UW.checkForEvent)
    Events.OnGameStart.Add(UW_ApplySandbox)
    Events.OnInitWorld.Add(UW_ApplySandbox)

    UW._eventsRegistered = true
    print("[UW] Eventos registrados (EveryTenMinutes + Sandbox).")
end

------------------------------------------------------------
-- Persistencia completa del evento, incluyendo overlays
------------------------------------------------------------

local UW_MD_KEY = "UW_ClimateEvent_v3"

local function UW_SaveStateFull()
    local md = ModData.getOrCreate(UW_MD_KEY)

    md.activeEventName    = UW.activeEventName
    md.eventEndWorldHours = UW.eventEndWorldHours
    md.currentEventType   = UW._currentEventType

    -- Guardar target climático actual
    md.targetClimate = UW._fadeInTargetClimate or UW._fadeStartClimate or {}

    -- Guardar overlays si existen
    md.overlayFog   = UW._fadeInTargetOverlayFog or nil
    md.overlayCloud = UW._fadeInTargetOverlayCld or nil

    ModData.transmit(UW_MD_KEY)

    print("[UW] [Persistencia] Estado completo guardado.")
end

local function UW_ClearStateFull()
    local md = ModData.getOrCreate(UW_MD_KEY)

    md.activeEventName    = nil
    md.eventEndWorldHours = nil
    md.currentEventType   = nil
    md.targetClimate      = nil
    md.overlayFog         = nil
    md.overlayCloud       = nil

    ModData.transmit(UW_MD_KEY)
    print("[UW] [Persistencia] Estado limpiado.")
end

------------------------------------------------------------
-- Hooks
------------------------------------------------------------

local _UW_startEvent_original = UW.startEvent
UW.startEvent = function(...)
    _UW_startEvent_original(...)
    if UW.activeEventName and UW.eventEndWorldHours then
        UW_SaveStateFull()
    end
end

local _UW_endEvent_original = UW.endEvent
UW.endEvent = function(...)
    _UW_endEvent_original(...)
    UW_ClearStateFull()
end


------------------------------------------------------------
-- Reconstrucción exacta del evento
------------------------------------------------------------

local function UW_ReapplyFullEvent(md)
    local cm = getWorld():getClimateManager()
    if not cm then
        print("[UW] No hay ClimateManager para reconstruir evento.")
        return
    end

    local target = md.targetClimate or {}

    -- Restaurar clima (override directo)
    local tempF = cm:getClimateFloat(4)
    local rainF = cm:getClimateFloat(3)
    local fogF  = cm:getClimateFloat(5)
    local windF = cm:getClimateFloat(6)
    local cldF  = cm:getClimateFloat(7)
    local snowF = cm:getClimateFloat(8)

    if tempF and target.temperature then
        tempF:setEnableOverride(true)
        tempF:setOverride(target.temperature, 1.0)
    end
    if rainF and target.rain then
        rainF:setEnableOverride(true)
        rainF:setOverride(target.rain, 1.0)
    end
    if fogF and target.fog then
        fogF:setEnableOverride(true)
        fogF:setOverride(target.fog, 1.0)
    end
    if windF and target.wind then
        windF:setEnableOverride(true)
        windF:setOverride(target.wind, 1.0)
    end
    if cldF and target.clouds then
        cldF:setEnableOverride(true)
        cldF:setOverride(target.clouds, 1.0)
    end
    if snowF and target.snow then
        snowF:setEnableOverride(true)
        snowF:setOverride(target.snow, 1.0)
    end

    -- Restaurar overlays si existían
    if md.overlayFog then
        local fogC = cm:getClimateColor(0)
        fogC:setEnableOverride(true)
        fogC:setOverride(md.overlayFog[1], md.overlayFog[2], md.overlayFog[3], md.overlayFog[4])
    end

    if md.overlayCloud then
        local cldC = cm:getClimateColor(1)
        cldC:setEnableOverride(true)
        cldC:setOverride(md.overlayCloud[1], md.overlayCloud[2], md.overlayCloud[3], md.overlayCloud[4])
    end

    -- Reactivar variables internas
    UW.activeEventName    = md.activeEventName
    UW.eventEndWorldHours = md.eventEndWorldHours
    UW._currentEventType  = md.currentEventType

    print("[UW] Evento reconstruido 1:1:", md.activeEventName)
end


------------------------------------------------------------
-- Restauración al cargar la partida
------------------------------------------------------------

local function UW_RestoreStateFull()
    local md = ModData.getOrCreate(UW_MD_KEY)

    if not md.activeEventName or not md.eventEndWorldHours then
        print("[UW] No hay evento previo para reconstruir.")
        return
    end

    local worldHours = getGameTime():getWorldAgeHours()

    print("[UW] Restaurando evento previo:", md.activeEventName)

    if worldHours < md.eventEndWorldHours then
        -- Evento sigue activo → reconstrucción exacta
        UW_ReapplyFullEvent(md)
    else
        -- Evento vencido → limpiez
        print("[UW] Evento expirado mientras no jugabas. Limpiando estado.")
        UW_ClearStateFull()
        UW.endEvent()
    end
end

Events.OnGameStart.Add(UW_RestoreStateFull)
Events.OnGameStart.Add(UW_RegisterEvents)
Events.OnInitWorld.Add(UW_RegisterEvents)
